<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Lock configuration class, used to get an instance of the currently configured lock factory.
 *
 * @package    core
 * @category   lock
 * @copyright  Damyon Wiese 2013
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace core\lock;

defined('MOODLE_INTERNAL') || die();

/**
 * Lock configuration class, used to get an instance of the currently configured lock factory.
 *
 * @package   core
 * @category  lock
 * @copyright Damyon Wiese 2013
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class lock_config {

    /**
     * Get an instance of the currently configured locking subclass.
     *
     * @param string $type - Unique namespace for the locks generated by this factory. e.g. core_cron
     * @return \core\lock\lock_factory
     * @throws \coding_exception
     */
    public static function get_lock_factory($type) {
        global $CFG, $DB;
        $lockfactory = null;

        if (during_initial_install()) {
            $lockfactory = new \core\lock\installation_lock_factory($type);
        } else if (isset($CFG->lock_factory) && $CFG->lock_factory != 'auto') {
            if (!class_exists($CFG->lock_factory)) {
                // In this case I guess it is not safe to continue. Different cluster nodes could end up using different locking
                // types because of an installation error.
                throw new \coding_exception('Lock factory set in $CFG does not exist: ' . $CFG->lock_factory);
            }
            $lockfactoryclass = $CFG->lock_factory;
            $lockfactory = new $lockfactoryclass($type);
        } else {
            $dbtype = clean_param($DB->get_dbfamily(), PARAM_ALPHA);

            // DB Specific lock factory is preferred - should support auto-release.
            $lockfactoryclass = "\\core\\lock\\${dbtype}_lock_factory";
            if (!class_exists($lockfactoryclass)) {
                $lockfactoryclass = '\core\lock\file_lock_factory';
            }
            /* @var lock_factory $lockfactory */
            $lockfactory = new $lockfactoryclass($type);
            if (!$lockfactory->is_available()) {
                // Final fallback - DB row locking.
                $lockfactory = new \core\lock\db_record_lock_factory($type);
            }
        }

        return $lockfactory;
    }

}
