<?php

/**
 * @defgroup cache
 */

/**
 * @file classes/cache/FileCache.inc.php
 *
 * Copyright (c) 2013-2016 Simon Fraser University Library
 * Copyright (c) 2000-2016 John Willinsky
 * Distributed under the GNU GPL v2. For full terms see the file docs/COPYING.
 *
 * @class FileCache
 * @ingroup cache
 *
 * @brief Provides caching based on machine-generated PHP code on the filesystem.
 */


import('lib.pkp.classes.cache.GenericCache');

class FileCache extends GenericCache {
	/**
	 * Connection to use for caching.
	 */
	var $filename;

	/**
	 * The cached data
	 */
	var $cache;

	/**
	 * Instantiate a cache.
	 */
	function FileCache($context, $cacheId, $fallback, $path) {
		parent::GenericCache($context, $cacheId, $fallback);

		$this->filename = $path . DIRECTORY_SEPARATOR . "fc-$context-" . str_replace('/', '.', $cacheId) . '.php';

		// Load the cache data if it exists.
		if (file_exists($this->filename)) {
			$this->cache = include($this->filename);
		} else {
			$this->cache = null;
		}
	}

	/**
	 * Flush the cache
	 */
	function flush() {
		unset($this->cache);
		$this->cache = null;
		if (file_exists($this->filename)) {
			unlink($this->filename);
		}
	}

	/**
	 * Get an object from the cache.
	 * @param $id
	 */
	function getCache($id) {
		if (!isset($this->cache)) return $this->cacheMiss;
		return (isset($this->cache[$id])?$this->cache[$id]:null);
	}

	/**
	 * Set an object in the cache. This function should be overridden
	 * by subclasses.
	 * @param $id
	 * @param $value
	 */
	function setCache($id, $value) {
		// Flush the cache; it will be regenerated on demand.
		$this->flush();
	}

	/**
	 * Set the entire contents of the cache.
	 */
	function setEntireCache(&$contents) {
		$newFile = !file_exists($this->filename);
		$fp = fopen($this->filename, 'wb');
		if ($newFile) {
			$umask = Config::getVar('files', 'umask');
			if ($umask) chmod($this->filename, FILE_MODE_MASK & ~$umask);
		}

		// If the cache can be written, write it. If not, fall
		// back on NO CACHING AT ALL.
		if ($fp) {
			fwrite ($fp, '<?php return ' . var_export($contents, true) . '; ?>');
			fclose ($fp);
		}
		$this->cache =& $contents;
	}

	/**
	 * Get the time at which the data was cached.
	 * If the file does not exist or an error occurs, null is returned.
	 * @return int
	 */
	function getCacheTime() {
		if (!file_exists($this->filename)) return null;
		$result = filemtime($this->filename);
		if ($result === false) return null;
		return ((int) $result);
	}

	/**
	 * Get the entire contents of the cache in an associative array.
	 */
	function &getContents() {
		if (!isset($this->cache)) {
			// Trigger a cache miss to load the cache.
			$this->get(null);
		}
		return $this->cache;
	}
}

?>
